<?php
/**
 * Main Player class.
 *
 * @package ctpl\classes
 */

namespace CTPL;

/**
 * Main Checker class.
 */
final class Player {
	/**
	 * Render WPS PLayer shortcode.
	 *
	 * @param string $content Content to search for WPS PLAYER shortcodes.
	 *
	 * @return string Content with WPS PLAYER shortcodes filtered out
	 */
	public function ctpl_render_shortcodes( $content ) {
		$regex = '/\[(.+)\]/m';
		preg_match_all( $regex, $content, $matches, PREG_SET_ORDER, 0 );

		// Print the entire match results.
		if ( is_array( $matches ) ) {
			foreach ( $matches as $shortcode ) {
				$shortcode_with_brackets    = $shortcode[0];
				$shortcode_without_brackets = $shortcode[1];
				$should_be_shortcode        = explode( ' ', $shortcode_without_brackets );
				$should_be_shortcode        = current( $should_be_shortcode );
				if ( shortcode_exists( $should_be_shortcode ) ) {
					$shortcode = do_shortcode( $shortcode_with_brackets );
					return $shortcode;
				}
			}
		}
		return $content;
	}

	/**
	 * Get all transformable iframe sources.
	 *
	 * @return array List of transformable iframe sources
	 */
	public static function get_transformable_sources() {
		return array(
			'redtube',
			'xvideos',
			'youporn',
			'youporngay',
		);
	}

	/**
	 * Create the WPS Player
	 *
	 * @param string                   $tag      The tag wrapped in the WPS Player. Can be <video>...</video> or <iframe>...</iframe>.
	 * @param string{'iframe','video'} $tag_type The type of tag to be rendered.
	 *
	 * @return string The WPS Player html code
	 */
	public static function create_wps_player( $tag, $tag_type ) {
		// Open wps-player tag.
		$wps_player = '<div class="wps-player">';
		// Add wps-player-block tag.
		$wps_player .= '<div class="wps-player-block">';
		// Add iframe wrapper tag.
		$wps_player .= 'iframe' === $tag_type ? '<div class="wps-player--iframe-wrapper">' : '';
		// Add <video> or <iframe> tag.
		$wps_player .= $tag;
		// Add Logo Watermark.
		$wps_player .= self::get_logo_watermark();
		// Add Desktop in-video ad start.
		$wps_player .= self::get_inside_player_desktop_ad_start( $tag_type );
		// Add Desktop in-video ad pause.
		$wps_player .= 'video' === $tag_type ? self::get_inside_player_desktop_ad_pause() : '';
		// Close iframe wrapper tag.
		$wps_player .= 'iframe' === $tag_type ? '</div>' : '';
		// Close wps-player tag.
		$wps_player .= '</div>';
		// Close wps-player-block tag.
		$wps_player .= '</div>';

		return $wps_player;
	}

	/**
	 * Get Logo Watermark.
	 *
	 * @return string The logo watermark tag if success || An empty string if not.
	 */
	public static function get_logo_watermark() {
		$logo_watermark_grayscale = 'on' === xbox_get_field_value( 'ctpl-options', 'logo-watermark-grayscale' ) ? 'wps-player__logo-watermark--grayscale' : '';
		$logo_watermark_file      = xbox_get_field_value( 'ctpl-options', 'image-logo-watermark-file' );
		$logo_watermark_position  = 'wps-player__logo-watermark--' . str_replace( ' ', '-', xbox_get_field_value( 'ctpl-options', 'logo-position-video-player' ) );
		$logo_watermark_width     = xbox_get_field_value( 'ctpl-options', 'logo-watermark-width' );
		$logo_watermark_opacity   = intval( xbox_get_field_value( 'ctpl-options', 'logo-watermark-opacity' ) ) / 100;

		if ( ( 'on' === xbox_get_field_value( 'ctpl-options', 'logo-watermark-video-player' ) ) && ( '' !== $logo_watermark_file ) ) {
			return '<img class="wps-player__logo-watermark ' . $logo_watermark_position . ' ' . $logo_watermark_grayscale . '" src="' . $logo_watermark_file . '" style="opacity: ' . $logo_watermark_opacity . '; width: ' . $logo_watermark_width . '%;" />';
		}
		return '';
	}

	/**
	 * Get inside player start ad.
	 *
	 * @param string{'iframe','video'} $tag to render the good button close text.
	 *
	 * @return string The Onpause Ad tag if not mobile and at least one ad is set || An empty string if not.
	 */
	public static function get_inside_player_desktop_ad_start( $tag ) {
		$ad_start_1       = trim( xbox_get_field_value( 'ctpl-options', 'inside-player-ad-zone-1-desktop' ) );
		$ad_start_2       = trim( xbox_get_field_value( 'ctpl-options', 'inside-player-ad-zone-2-desktop' ) );
		$video_close_text = xbox_get_field_value( 'ctpl-options', 'ctpl-button-close-video-label' );
		if ( empty( $video_close_text ) ) {
			$video_close_text = __( 'Close and play', 'ctpl_lang' );
		}
		$iframe_close_text = xbox_get_field_value( 'ctpl-options', 'ctpl-button-close-iframe-label' );
		if ( empty( $iframe_close_text ) ) {
			$iframe_close_text = __( 'Close', 'ctpl_lang' );
		}
		if ( ( empty( $ad_start_1 ) && empty( $ad_start_2 ) ) ) {
			return false;
		}
		$close_btn_text = 'video' === $tag ? $video_close_text : $iframe_close_text;
		$style_display  = 'video' === $tag ? 'none' : 'flex';
		return '
		<div class="wps-player__happy-inside wps-player__happy-inside--start" style="display:' . $style_display . ';">
			<div class="wps-player__happy-inside-zone">
				<div class="wps-player__happy-inside-1">' . do_shortcode( $ad_start_1 ) . '</div>
				<div class="wps-player__happy-inside-2">' . do_shortcode( $ad_start_2 ) . '</div>
			</div>
			<button class="wps-player__happy-inside-btn-close">' . $close_btn_text . '</button>
		</div>';
	}

	/**
	 * Get inside player pause ad.
	 *
	 * @return string The Onpause Ad tag if not mobile and at least one ad is set || An empty string if not.
	 */
	public static function get_inside_player_desktop_ad_pause() {
		$ad_pause_1 = trim( xbox_get_field_value( 'ctpl-options', 'vp-ad-pause-1' ) );
		$ad_pause_2 = trim( xbox_get_field_value( 'ctpl-options', 'vp-ad-pause-2' ) );

		if ( ( empty( $ad_pause_1 ) && empty( $ad_pause_2 ) ) ) {
			return false;
		}

		return '
		<div class="wps-player__happy-inside wps-player__happy-inside--pause" style="display:none;">
			<div class="wps-player__happy-inside-zone">
				<div class="wps-player__happy-inside-1">' . do_shortcode( $ad_pause_1 ) . '</div>
				<div class="wps-player__happy-inside-2">' . do_shortcode( $ad_pause_2 ) . '</div>
			</div>
			<button class="wps-player__happy-inside-btn-close">' . __( 'Close and play', 'ctpl_lang' ) . '</button>
		</div>';
	}

	/**
	 * Get video type from a given video url.
	 *
	 * @param string|int $resolution The resolution.
	 *
	 * @return bool true if the given resolution is an HD resolution, false if not
	 */
	public static function is_hd_resolution( $resolution ) {
		$hd_resolutions = array( 8, 4320, 4, 2160, 1080, 720 );
		return in_array( intval( $resolution ), $hd_resolutions, true );
	}

	/**
	 * Get video type from a given video url.
	 *
	 * @param string $video_url The Video url.
	 *
	 * @return string The video format detected
	 */
	public static function get_type_from_video_url( $video_url ) {
		// Get allowed mime types.
		$allowed_mime_types = get_allowed_mime_types();
		// And add m3u8 type.
		$allowed_mime_types['m3u8'] = 'application/x-mpegURL';
		// Retrieve video type.
		$parsed_video_url = wp_parse_url( $video_url );
		$video_file_name  = explode( '/', $parsed_video_url['path'] );
		$video_file_name  = end( $video_file_name );
		$video_format     = wp_check_filetype( $video_file_name, $allowed_mime_types );
		$video_type       = $video_format['type'];
		return $video_type;
	}

	/**
	 * Makes brilliant cUrl calls.
	 *
	 * @param string $url          The array to clear.
	 * @param string $referer      The array to clear.
	 * @param string $device_type  The device type {'Desktop', Mobile'}.
	 * @param array  $cookies_list The array to clear.
	 * @return string The content from curl call.
	 */
	public static function curl( $url = '', $referer = '', $device_type = 'Desktop', $cookies_list = array() ) {
		$agent                 = \Campo\UserAgent::random( array( 'device_type' => $device_type ) );
		$custom_proxy_ip       = xbox_get_field_value( 'ctpl-options', 'ctpl-proxy-ip' );
		$custom_proxy_port     = xbox_get_field_value( 'ctpl-options', 'ctpl-proxy-port' );
		$custom_proxy_user     = xbox_get_field_value( 'ctpl-options', 'ctpl-proxy-user' );
		$custom_proxy_password = xbox_get_field_value( 'ctpl-options', 'ctpl-proxy-password' );

		$cookie_file = CTPL_DIR . 'cookie.txt';

		// phpcs:disable
		$ch = curl_init( $url );
		if ( ! empty( $custom_proxy_ip ) && ! empty( $custom_proxy_port ) ) {
			curl_setopt( $ch, CURLOPT_PROXY, $custom_proxy_ip );
			curl_setopt( $ch, CURLOPT_PROXYPORT, $custom_proxy_port );
			if ( ! empty( $custom_proxy_user ) && ! empty( $custom_proxy_password ) ) {
				curl_setopt( $ch, CURLOPT_PROXYUSERPWD, $custom_proxy_user . ':' . $custom_proxy_password );
			}
		}
		if ( ! empty( $cookies_list ) ) {
			$cookies = implode( '; ', $cookies_list );
			curl_setopt( $ch, CURLOPT_COOKIE, $cookies );
		}
		curl_setopt( $ch, CURLOPT_USERAGENT, $agent );
		curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
		curl_setopt( $ch, CURLOPT_TIMEOUT, 3 );
		curl_setopt( $ch, CURLOPT_COOKIESESSION, true );
		curl_setopt( $ch, CURLOPT_REFERER, $referer );
		curl_setopt( $ch, CURLOPT_SSL_VERIFYPEER, false);
		curl_setopt( $ch, CURLOPT_FOLLOWLOCATION, true );
		curl_setopt( $ch, CURLOPT_AUTOREFERER, true );

		curl_setopt( $ch, CURLOPT_COOKIEFILE, $cookie_file );
		curl_setopt( $ch, CURLOPT_COOKIEJAR, $cookie_file );

		$page = curl_exec( $ch );
		curl_close( $ch );
		// phpcs:enable
		return $page;
	}

	/**
	 * Find substring in a given string
	 *
	 * @param string $str_start The start string.
	 * @param string $str_end   The end string.
	 * @param string $str       The string in which to search for the substring.
	 * @return string The string to find, empty string if nothing is found.
	 */
	public static function find_string( $str_start, $str_end, $str ) {
		$regex = '';
		// phpcs:disable
		eval( WPSCORE()->eval_product_data( 'CTPL', 'find_string_regex' ) );
		// phpcs:enable
		preg_match( $regex, $str, $matches, PREG_OFFSET_CAPTURE, 0 );
		if ( isset( $matches[1] ) && isset( $matches[1][0] ) ) {
			return $matches[1][0];
		}
		return '';
	}

	/**
	 * Find param value in a given string.
	 *
	 * @param string $param   	The para to search the value in (ie "src", "class", "id")
	 * @param string $str       The string in which to search for the param value.
	 * @return string The string to find, empty string if nothing is found.
	 */
	public static function find_html_param_value( $param, $str ) {
		$str = strtolower( $str );
		$value = '';
		$value = self::find_string( $param . '="', '"', $str );
		if ( '' === $value ) {
			$value = self::find_string( $param . '=', ' ', $str );
		}
		if ( '' === $value ) {
			$value = self::find_string( $param . '=', '/>', $str );
		}
		if ( '' === $value ) {
			$value = self::find_string( $param . '=', '>', $str );
		}
		return trim( $value );
	}

	/**
	 * Set the list of all terms to translate
	 *
	 * @return array List of all terms to translate
	 */
	public function get_object_l10n() {
		return array();
	}
}

