<?php
/**
 * Dom manipulation class.
 *
 * @package ctpl\classes
 */

namespace CTPL;

/**
 * Dom manipulation class.
 */
final class Dom {
	/**
	 * Prepare <iframe> and <video> blocks
	 *
	 * @see prepare_iframes( $content ) & prepare_videos( $content )
	 *
	 * @param string $content The page content passed to the function.
	 *
	 * @return string The HTML result.
	 */
	public function prepare_iframes_and_videos( $content ) {
		$content = $this->prepare_iframes( $content );
		$content = $this->prepare_videos( $content );
		return $content;
	}

	/**
	 * Prepend loader block add transform-iframe class to iframes.
	 *
	 * @param string $content The page content passed to the function.
	 *
	 * @return string The HTML result
	 */
	public function prepare_iframes( $content ) {
		if ( 'off' === xbox_get_field_value( 'ctpl-options', 'transform-iframe-video-player' ) ) {
			return $content;
		}
		global $post;
		$post_id = is_single() ? intval( $post->ID ) : 0;
		preg_match_all( '/(?:<iframe[^>]*)(?:(?:\/>)|(?:>.*?<\/iframe>))/', $content, $iframes, PREG_SET_ORDER );
		foreach ( (array) $iframes as $iframe ) {

			if ( empty( $iframe ) ) {
				continue;
			}

			$iframe         = $iframe[0];
			$iframe_classes = Player::find_html_param_value( 'class', $iframe );
			$iframe_src     = Player::find_html_param_value( 'src', $iframe );

			// Bypass WP-Script themes wpst-trailers <video> tags.
			if ( false !== strpos( $iframe_classes, 'wpst-trailer' ) ) {
				continue;
			}

			// Bypass classes from options to exclude.
			if ( array_intersect( explode( ' ', $iframe_classes ), $this->get_classes_to_bypass() ) ) {
				continue;
			}

			// Bypass ids from options to exclude.
			$iframe_id = Player::find_html_param_value( 'id', $iframe );

			if ( array_intersect( explode( ' ', $iframe_id ), $this->get_ids_to_bypass() ) ) {
				continue;
			}

			// Bypass iframes that must not be rendered with CTPL iframe.
			if ( ! $this->is_compatible_source_url( $iframe_src ) ) {
				continue;
			}
			$iframe_tag = rawurlencode( $iframe );
			$new_iframe = $this->render_ctpl_iframe( 'iframe', $iframe_tag, $post_id );
			$content    = str_replace( $iframe, $new_iframe, $content );
		}
		return $content;
	}

	/**
	 * Prepend loader block add transform-iframe class to iframes.
	 *
	 * @param string $content The page content passed to the function.
	 *
	 * @return string The HTML result
	 */
	public function prepare_videos( $content ) {
		if ( 'off' === xbox_get_field_value( 'ctpl-options', 'transform-basic-video-player' ) ) {
			return $content;
		}
		global $post;
		$post_id = is_single() ? intval( $post->ID ) : 0;
		preg_match_all( '/(?:<video[^>]*)(?:(?:\/>)|(?:>.*?<\/video>))/', $content, $videos, PREG_SET_ORDER );
		foreach ( (array) $videos as $video ) {

			if ( empty( $video ) ) {
				continue;
			}

			$video         = $video[0];
			$video_classes = Player::find_html_param_value( 'class', $video );

			// Bypass WP-Script themes wpst-trailers <video> tags.
			if ( false !== strpos( $video_classes, 'wpst-trailer' ) ) {
				continue;
			}

			// Bypass classes from options to exclude.
			if ( array_intersect( explode( ' ', $video_classes ), $this->get_classes_to_bypass() ) ) {
				continue;
			}

			// Bypass ids from options to exclude.
			$video_id = Player::find_html_param_value( 'id', $video );

			if ( array_intersect( explode( ' ', $video_id ), $this->get_ids_to_bypass() ) ) {
				continue;
			}

			$video_tag = rawurlencode( $video );
			$new_video = $this->render_ctpl_iframe( 'video', $video_tag, $post_id );
			$content   = str_replace( $video, $new_video, $content );
		}
		return $content;
	}

	/**
	 * Get all ids to bypass from the options.
	 *
	 * @return array The array of ids to bypass.
	 */
	private function get_ids_to_bypass() {
		$regex             = '/\#([a-zA-Z0-9-_]+)/m';
		$classes_to_bypass = xbox_get_field_value( 'ctpl-options', 'bypass-transform' );
		preg_match_all( $regex, $classes_to_bypass, $matches, PREG_SET_ORDER, 0 );
		$ids = array_map(
			function( $match ) {
				return $match[1];
			},
			$matches
		);
		return $ids;
	}

	/**
	 * Get all classes to bypass from the options.
	 *
	 * @return array The array of classes to bypass.
	 */
	private function get_classes_to_bypass() {
		$regex             = '/\.([a-zA-Z0-9-_]+)/m';
		$classes_to_bypass = xbox_get_field_value( 'ctpl-options', 'bypass-transform' );
		preg_match_all( $regex, $classes_to_bypass, $matches, PREG_SET_ORDER, 0 );
		$class_names = array_map(
			function( $match ) {
				return $match[1];
			},
			$matches
		);
		return $class_names;
	}

	/**
	 * Check if a given url is included in the compatible sources array.
	 *
	 * @param string $url The url to test.
	 *
	 * @return string|bool string source name if found, bool false if not
	 */
	private function is_compatible_source_url( $url ) {
		foreach ( $this->get_compatible_iframe_sources() as $source ) {
			if ( false !== strpos( $url, $source ) ) {
				return $source;
			}
		}
		return false;
	}

	/**
	 * Get all iframe sources managed by WPS Player
	 *
	 * @see get_transformable_sources() & get_not_transformable_sources()
	 *
	 * @return array List of available iframe sources
	 */
	public function get_compatible_iframe_sources() {
		return array_merge( $this->get_transformable_sources(), $this->get_not_transformable_sources() );
	}

	/**
	 * Get all transformable iframe sources
	 *
	 * @return array List of transformable iframe sources
	 */
	public function get_transformable_sources() {
		return array(
			'redtube',
			'xvideos',
			'youporn',
			'youporngay',
		);
	}

	/**
	 * Get all not transformable iframe sources
	 *
	 * @return array List of not transformable iframe sources
	 */
	public function get_not_transformable_sources() {
		return array(
			'drive.google',
			'eporner',
			'openload',
			'pornhub',
			'vimeo',
			'xhamster',
			'youtube',
		);
	}

	/**
	 * Render CTPL Iframe.
	 *
	 * @param string $type The type of media to render (iframe or video).
	 * @param string $tag  The html tag of the media.
	 * @param int    $post_id The post id where the media has been found. 0 if no post.
	 *
	 * @return string The CTPL iframe tag to render in the page.
	 */
	public function render_ctpl_iframe( $type, $tag, $post_id ) {
		$query      = base64_encode( 'post_id=' . $post_id . '&type=' . $type . '&tag=' . $tag );
		$iframe_url = CTPL_URL . "public/player-x.php?q=$query";
		if ( ! CTPL()->is_wps_theme() && 'yes' === xbox_get_field_value( 'ctpl-options', 'responsive-player' ) ) {
			$iframe  = '<div style="position:relative;overflow:hidden;padding-top:56.25%;">';
			$iframe .= '<iframe src="' . $iframe_url . '" frameborder="0" scrolling="no" allowfullscreen style="position: absolute;top: 0;left: 0;width: 100%;height: 100%;border: 0;"></iframe>';
			$iframe .= '</div>';
		} else {
			$iframe = '<iframe src="' . $iframe_url . '" frameborder="0" scrolling="no" allowfullscreen></iframe>';
		}
		return $iframe;
	}
}

