<?php
/**
 * Plugin class.
 *
 * @package ctpl\classes
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * CTPL Plugin Singleton Class.
 */
final class CTPL {
	/**
	 * The instance of WPS PLAYER plugin
	 *
	 * @var instanceof CTPL $instance
	 * @access private
	 * @static
	 */
	private static $instance;

	/**
	 * The config of WPS PLAYER plugin
	 *
	 * @var array $config
	 * @access private
	 * @static
	 */
	private static $config;

	/**
	 * __clone method
	 *
	 * @return void
	 */
	public function __clone() {
		_doing_it_wrong( __FUNCTION__, 'Cannot clone ' . __CLASS__, '1.0' );}

	/**
	 * __wakeup method
	 *
	 * @return void
	 */
	public function __wakeup() {
		_doing_it_wrong( __FUNCTION__, 'Cannot wakeup ' . __CLASS__, '1.0' );}

	/**
	 * Instance method
	 *
	 * @return self::$instance
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof CTPL ) ) {
			include_once ABSPATH . 'wp-admin/includes/plugin.php';
			self::$instance = new CTPL();
			// Load textdomain.
			self::$instance->load_textdomain();
			// load config file.
			require_once CTPL_DIR . 'config.php';
			// load options file.
			require_once CTPL_DIR . 'admin/pages/page-options-x.php';
			if ( is_admin() ) {
				self::$instance->load_admin_filters();
				self::$instance->load_admin_hooks();
				self::$instance->auto_load_php_files( 'admin' );
				self::$instance->admin_init();
			}
			if ( ! is_admin() ) {
				self::$instance->auto_load_php_files( 'public' );
				self::$instance->public_init();
			}
		}
		return self::$instance;
	}

	/**
	 * Add js and css files, tabs, pages, php files in admin mode.
	 *
	 * @return void
	 */
	public function load_admin_filters() {
		add_filter( 'WPSCORE-scripts', array( $this, 'add_admin_scripts' ) );
		add_filter( 'WPSCORE-tabs', array( $this, 'add_admin_navigation' ) );
		add_filter( 'WPSCORE-pages', array( $this, 'add_admin_navigation' ) );
	}

	/**
	 * Add admin js and css scripts. This is a WPSCORE-scripts filter callback function.
	 *
	 * @param array $scripts List of all WPS CORE CSS / JS to load.
	 *
	 * @return array $scripts List of all WPS CORE + WPS PLAYER CSS / JS to load.
	 */
	public function add_admin_scripts( $scripts ) {
		if ( isset( self::$config['scripts'] ) ) {

			if ( isset( self::$config['scripts']['js'] ) ) {
				$scripts += (array) self::$config['scripts']['js'];
			}
			if ( isset( self::$config['scripts']['css'] ) ) {
				$scripts += (array) self::$config['scripts']['css'];
			}
		}
		return $scripts;
	}

	/**
	 * Add plugin admin navigation tab. This is a WPSCORE-tabs and WPSCORE-pages filters callback function.
	 *
	 * @param array $nav List of all WPS CORE navigation tabs to add.
	 *
	 * @return array $nav List of all WPS CORE + WPS PLAYER navigation tabs to add.
	 */
	public function add_admin_navigation( $nav ) {
		if ( isset( self::$config['nav'] ) ) {
		// phpcs:disable
			eval( WPSCORE()->eval_product_data( 'CTPL', 'add_admin_navigation' ) );
			// phpcs:enable
		}
		return $nav;
	}

	/**
	 * Auto-loader for PHP files
	 *
	 * @param string{'admin','public'} $dir Directory where to find PHP files to load.
	 * @static
	 * @return void
	 */
	public static function auto_load_php_files( $dir ) {
		$dirs = (array) ( CTPL_DIR . $dir . '/' );
		foreach ( (array) $dirs as $dir ) {
			$files = new RecursiveIteratorIterator( new RecursiveDirectoryIterator( $dir ) );
			if ( ! empty( $files ) ) {
				foreach ( $files as $file ) {
					// exlude dir.
					if ( $file->isDir() ) {
						continue; }
					// exlude index.php.
					if ( $file->getPathname() === 'index.php' ) {
						continue; }
					// exlude files != .php.
					if ( substr( $file->getPathname(), -4 ) !== '.php' ) {
						continue; }
					// exlude files from -x suffixed directories.
					if ( substr( $file->getPath(), -2 ) === '-x' ) {
						continue; }
					// exlude -x suffixed files.
					if ( substr( $file->getPathname(), -6 ) === '-x.php' ) {
						continue; }
					// else require file.
					require $file->getPathname();
				}
			}
		}
	}

	/**
	 * Registering plugin activation / deactivation / uninstall hooks.
	 *
	 * @return void
	 */
	public function load_admin_hooks() {
		register_activation_hook( __FILE__, array( 'CTPL', 'activation' ) );
		register_deactivation_hook( __FILE__, array( 'CTPL', 'deactivation' ) );
		register_uninstall_hook( __FILE__, array( 'CTPL', 'uninstall' ) );
	}

	/**
	 * Stuff to do on plugin activation. This is a register_activation_hook callback function.
	 *
	 * @static
	 *
	 * @return void
	 */
	public static function activation() {
		WPSCORE()->update_client_signature();
		WPSCORE()->init( true );
	}

	/**
	 * Stuff to do on plugin deactivation. This is a register_deactivation_hook callback function.
	 *
	 * @static
	 *
	 * @return void
	 */
	public static function deactivation() {
		WPSCORE()->update_client_signature();
		WPSCORE()->init( true );
	}

	/**
	 * Stuff to do on plugin deactivation. This is a register_deactivation_hook callback function.
	 *
	 * @static
	 *
	 * @return void
	 */
	public static function uninstall() {
		WPSCORE()->update_client_signature();
		WPSCORE()->init( true );
	}

	/**
	 * Load textdomain method.
	 *
	 * @return bool True when textdomain is successfully loaded, false if not.
	 */
	public function load_textdomain() {
		$lang = ( current( explode( '_', get_locale() ) ) );
		if ( 'zh' === $lang ) {
			$lang = 'zh-TW';
		}
		$textdomain = 'ctpl_lang';
		$mofile     = CTPL_DIR . "languages/{$textdomain}_{$lang}.mo";
		return load_textdomain( $textdomain, $mofile );
	}

	/**
	 * Stuff to do on admin init.
	 *
	 * @return void
	 */
	private function admin_init() {}

	/**
	 * Stuff to do on public init.
	 *
	 * @access private
	 * @return void
	 */
	private function public_init() {
		add_action( 'wp_head', array( $this, 'ctpl_head' ), 9999 );
	}

	/**
	 * Stuff to do before rendering the page.
	 *
	 * @return void
	 */
	public function ctpl_head() {
		$ctpl_dom = new CTPL\Dom();
		ob_start( array( $ctpl_dom, 'prepare_iframes_and_videos' ) );
	}

	/**
	 * Check if current theme is a theme from WP-Script.
	 *
	 * @return bool true if current theme is a theme from WP-Script, false if not
	 */
	public static function is_wps_theme() {
		$current_theme = wp_get_theme();
		return 'WP-Script' === $current_theme->get( 'Author' );
	}

}
